import MaterialEditorService from "../services/MaterialEditorService.js";

class ParamIter {
    _data;
    _dataKey;
    _children;
    _arrayValues;
    _controller;
    name;
    desc;

    constructor(desc, data, dataKey, name, controller) {
        this.desc = desc;
        
        // data is indirected to allow us to set value types
        this._data = data;
        this._dataKey = dataKey;
        this.name = name;
        this._controller = controller;
    }	
    ArrayValues() {
        if( null != this._arrayValues )
            return this._arrayValues;
        
        this._arrayValues  = [];
        for(var i = 0; i < this.desc.arraySize; ++i)
            this._arrayValues[i] = new ParamIter( this.desc, this._data[this._dataKey], i, this.name + "[" + i + "]", this._controller );
        return this._arrayValues;
    }
    Children() {
        if( null != this._children )
            return this._children;
        
        this._children = {};
        var schemaArray = this._controller.GetSchema( this.desc.type );        
        for(var i in schemaArray) {
            var key = schemaArray[i].name;
            this._children[key] = new ParamIter( schemaArray[i], this._data[this._dataKey], key, key, this._controller );
        }
        return this._children;
    }
    HasToolTip() {
        return "toolTip" in this.desc.edit;
    }
    GetToolTip() {
        return this.desc.edit.toolTip;
    }
    EditMin() {
        return this.desc.edit.min;
    }
    EditMax() {
        return this.desc.edit.max;
    }
    get numberValue() {
        return this._data[this._dataKey];
    }
    set numberValue(val) {
        this._data[this._dataKey] = Number(val);
        this._controller.UpdateMaterial();
    }
    get numberValue0() {
        return this._data[this._dataKey][0];
    }
    set numberValue0(val) {
        this._data[this._dataKey][0] = Number(val);
        this._controller.UpdateMaterial();
    }
    get numberValue1() {
        return this._data[this._dataKey][1];
    }
    set numberValue1(val) {
        this._data[this._dataKey][1] = Number(val);
        this._controller.UpdateMaterial();
    }
    get colourValue() {
        var result = this._controller.ArrayToColor(this._data[this._dataKey]);
        return result;
    }
    set colourValue(val) {
        this._data[this._dataKey] = this._controller.ColorToArray(val);
        this._controller.UpdateMaterial();
    }
}

class TextureWrapper {
    _controller;
    _textureDetails;
    _flags = {};
    _filterFlags = {"Anisotropic" : "CUBETF_ANISOTROPIC", "Trilinear" : "CUBETF_TRILINEAR", "Default" : null};
    _uriUnique;
    constructor(texture, controller) {
        this._controller = controller;
        this._textureDetails = texture;
        for(var i in texture.Flags) {
            this._flags[ texture.Flags[i] ] = 1;
        }
        this._uriUnique = (1e9 * Math.random()).toFixed();
    }
    get Name() {
        return this._textureDetails.Name;
    }
    Resource() {
        return this._controller.FileName( this._textureDetails.Texture );
    }
    Type() {
        // remove the enum prefix
        return this._textureDetails.StageType.toLowerCase().substring(8);
    }
    Full() {
        return this._textureDetails;
    }
    ToggleFlag(flag) {
        if(this.HasFlag(flag))
            this.RemFlag(flag);
        else
            this.AddFlag(flag);

        this._controller.ForceUpdateMaterial();
    }
    HasFlag(flag) {
        return flag in this._flags;
    }
    RemFlag(flag) {
        if( this.HasFlag( flag ) ) {
            delete this._flags[flag];
            this._textureDetails.Flags = Object.keys( this._flags );
            console.log(this._textureDetails.Flags);
        }
    }
    AddFlag(flag) {
        if( null == flag )
            return;

        if( !this.HasFlag( flag ) ) {
            this._flags[flag] = 1;
            this._textureDetails.Flags = Object.keys( this._flags );
            console.log(this._textureDetails.Flags);
        }
    }
    get Filter() {
        for( var flag in this._filterFlags ) {
            var gameFlag = this._filterFlags[flag];

            // default to the null flag, note this should be last in the array
            if( gameFlag == null || this.HasFlag(gameFlag) )
                return flag;
        }
    }
    set Filter(val) {
        if( val != this.Filter ) {
            for( var flag in this._filterFlags ) {
                var gameFlag = this._filterFlags[flag];
                this.RemFlag(gameFlag);
            }
            this.AddFlag( this._filterFlags[val] );
        }

        this._controller.ForceUpdateMaterial();
    }
    get FilterModes() {
        return Object.keys(this._filterFlags);
    }
    get URI() {
        return this._controller.materialEditorService.MaterialEditorUrl 
            + this._controller.MaterialID() 
            +"/" + this.Name + "?"+ "R="
            + this._uriUnique;
    }
    SetDefault() {
        this.Texture = '';

        // this is not a good solution this function should just clear the file path however I couldn't find an easy way to make that work
        location.reload(true);
    }
    set Texture(val) {
        this._textureDetails.Texture = val.replace(/^.*[\\\/]/, '');
        var scopeThis = this;
        this._controller.ForceUpdateMaterial().then( function() {
            scopeThis._uriUnique = (1e9 * Math.random()).toFixed();
        })
    }

    get Chroma() {
        return this.HasFlag("CUBETF_CHROMAKEY");
    }
    set Chroma(val) {
        this.ToggleFlag("CUBETF_CHROMAKEY");
    }

    get NoReducedDetail() {
        return this.HasFlag("CUBETF_NOREDUCEDETAIL");
    }
    set NoReducedDetail(val) {
        this.ToggleFlag("CUBETF_NOREDUCEDETAIL");
    }
}

class MacroWrapper {    
    _controller;
    _macroDetails;
    _currentSetting;
    _pendingSetting;
    DropDownValues = [];

    constructor(macro, controller, currentSetting) {
        this._controller = controller;
        this._macroDetails = macro;
        this._currentSetting = currentSetting == null ? macro.default : currentSetting;
        this._pendingSetting = this._currentSetting;

        var dropDownValues = [];
        macro.values.forEach( function(val) {
            var v = {};
            v["macro"] = val;
            if(val == "@Undef")
                v["ui"] = "Disable";
            else if(val == "@Def")
                v["ui"] = "Enable";
            else if(val == "@Null")
                v["ui"] = "Enable";
            else v["ui"] = val;
            dropDownValues.push( v );
        });
        this.DropDownValues = dropDownValues;
    }
    GetUIName() {
        return this._macroDetails.ui_name;
    }
    IsCheckboxMacro() {
        return 2 == this._macroDetails.values.length
            && this._macroDetails.values.includes("@Undef")
            && this._macroDetails.values.includes("@Null");
    }
    AppendLineBreak() {
        if( "break" in this._macroDetails )
            return this._macroDetails.break == 1;
        return false;
    }
    HasToolTip() {
        return "toolTip" in this._macroDetails;
    }
    GetToolTip() {
        return this._macroDetails.toolTip;
    }
    get Name() {
        return this._macroDetails.macro;
    }
    get Value() {
        return this._pendingSetting;
    }
    set Value(val) {
        var wasDirty = this._pendingSetting != this._currentSetting;
        this._pendingSetting = val;
        var isDirty = this._pendingSetting != this._currentSetting;
        if(wasDirty != isDirty)
            this._controller.FlagsDirty += wasDirty ? -1 : 1;
    }
    get Flag() {
        return this._pendingSetting == "@Null";
    }
    set Flag(val) {
        this.Value = val ? "@Null" : "@Undef";
    }
}

export default class MaterialEditorCtrl {
    _materialDetails;
    Parameters = [];
    Macros = [];
    Textures = [];
    EditableTextures = [];
    FlagsDirty = 0;
    _materialId;
    _lastMaterialDispatch;
    _updateId = null;
    _searchPaths;

    _blendFlags = {"Zero" : "CUBEAB_ZERO", "One" : "CUBEAB_ONE", "SrcColor" : "CUBEAB_SRCCOLOR", "InvSrcColor" : "CUBEAB_INVSRCCOLOR", "SrcAlpha" : "CUBEAB_SRCALPHA", "InvSrcAlpha" : "CUBEAB_INVSRCALPHA", "DstAlpha" : "CUBEAB_DSTALPHA", "InvDstAlpha" : "CUBEAB_INVDSTALPHA", "DstColor" : "CUBEAB_DSTCOLOR", "InvDstColor" : "CUBEAB_INVDSTCOLOR", "SrcAlphaSat" : "CUBEAB_SRCALPHASAT" };
    materialEditorService;

    static $inject = ['materialDetails', 'materialEditorService', '$stateParams', '$state'];
    static resolve = {
        materialDetails: ['materialEditorService', '$stateParams', (materialEditorService, $stateParams) => materialEditorService.getMaterialDetails($stateParams['materialId'])]
    };
    constructor(materialDetails, materialEditorService, $stateParams, $state) {
        this.materialEditorService = materialEditorService;
        this._materialDetails = materialDetails;
        this._materialId = $stateParams['materialId'];

        var matCtrl = this;

        // initialise our parameter iterators
		var desc = this._materialDetails.schema.params;
        var keys = Object.keys( desc );
		for( var key in keys ) {
			var type = desc[key];
			var t = {
				"type" : type,
				"arraySize" : -1,
				"edit" : {}				
			}
            this.Parameters.push( new ParamIter( t, this._materialDetails.values.parameters, type, type, this ) )
        }

        var textures = this._materialDetails.values.textures;
		for(var idx in textures) {
            var wrapper = new TextureWrapper(textures[idx], matCtrl);
            this.Textures.push( wrapper );

            // Editable textures are filtered by those which can actually be edited at runtime
            if( !textures[idx].inactive && !textures[idx].runtime && textures[idx].Name[0] != ':')
                this.EditableTextures.push( wrapper );
        }

        var macroSettings = this._materialDetails.values.preprocessorFlags;
        if(null != this._materialDetails.schema.macros) {
            this._materialDetails.schema.macros.forEach(function(macro) {
                var initialValue = null;
                if(null != macroSettings && macro.macro in macroSettings)
                    initialValue = macroSettings[macro.macro];
                matCtrl.Macros.push(new MacroWrapper(macro, matCtrl, initialValue));
            });
        }

        this._searchPaths = JSON.parse(JSON.stringify(this._materialDetails.meta.searchPaths));
        for(var pathIndex in this._searchPaths)  {
            this._searchPaths[pathIndex] = this.PreparePath(this._searchPaths[pathIndex]);
        }
    }
    MaterialID() {
        return this._materialId;
    }
    PreparePath(value) {
        var path = value;
        
        // normalize
        path = path.replace("\\\\", "\\");
        
        // cut off our root
        var pos = path.indexOf("MODDEV");
        if(pos != -1)
            return path.substring(pos + 7);
        return path;
    }
    FileName(path) {
        while(-1 != path.indexOf('\\'))
            path = path.substring(path.indexOf('\\') + 1);
        return path;
    }
    Name() {
        return this._materialDetails.meta.name;
    }
    get Shader() {
        return this._materialDetails.meta.shader;
    }
    set Shader(val) {
        var scopeThis = this;
        this.materialEditorService.setMaterialShader(this._materialId, val).then(function() {
            location.reload(true); // if the shader was changed we need to reload everything
        })
    }
    GetShaderOptions() {
        return this._materialDetails.meta.shaderOptions;
    }
    get Mapper() {
        return this._materialDetails.values.refName;
    }
    set Mapper(val) {
        this._materialDetails.values.refName = val;
        this.ForceUpdateMaterial();
    }
    GetMapperOptions() {
        return this._materialDetails.meta.mapperOptions;
    }
    VertexShader() {
        return this._materialDetails.meta.vertexShader;
    }
    PixelShader() {
        return this._materialDetails.meta.pixelShader;
    }
    SearchPaths() {
        return this._searchPaths;
    }
    SourcePath() {
        return this.PreparePath(this._materialDetails.meta.path);
    }
	ColorToArray(v) {
		var r = parseInt( v.slice(1,3), 16 ) / 255.0
		var g = parseInt( v.slice(3,5), 16 ) / 255.0
		var b = parseInt( v.slice(5), 16 ) / 255.0
		return [r, g, b, 1.0]
	}
	ValueToByteStr(v) {
		var result = "" + Math.round( Math.min(v * 255,255) ).toString(16)
		while(result.length < 2) {
            result = '0' + result;
        }
		return result;
    }
    UpdateMaterial() {
        var d = new Date();
        var n = d.getTime();
        var dispatchLimitMilliseconds = 100;
        
        // don't spam too many updates but ensure we schedule a final update if we can't update now
        var nextDispatch = this._lastMaterialDispatch + dispatchLimitMilliseconds - n;
        if( nextDispatch > 0 ) {

            // if we don't have a pending async update lets schedule one
            if( null == this._updateId ) {
                var _this = this;
                this._updateId = setTimeout(function() {
                    _this.ForceUpdateMaterial();
                    _this._updateId = null;
                }, 300); // 300ms is an arbitary time that prevents this from firing between calls to UpdateMaterial while dragging a slider about. 
                         // I found setting this much lower resulted in the callback firing before it was cancelled causing spam.
            }
            return;
        }

        // JS is generally single threaded so this code is robust
        if(null != this._updateId) {
            clearTimeout(this._updateId);
            this._updateId = null;
        }
        this._lastMaterialDispatch = n;
        this.materialEditorService.setMaterialDetails(this._materialId, JSON.stringify(this._materialDetails.values));
    }
    ForceUpdateMaterial() {
        return this.materialEditorService.setMaterialDetails(this._materialId, JSON.stringify(this._materialDetails.values));
    }
    UploadMaterial(f) {
        var _this = this;
        var reader = new FileReader();

        // Handle progress, success, and errors
        reader.onload = function(x) {
            _this.materialEditorService.setMaterialDetails(_this._materialId, reader.result);

            // when we upload a material we need to do a full reload
            location.reload(true);
        }
        // Read file into memory as UTF-8
        reader.readAsText(f.target.files[0], "UTF-8");
    }
    TriggerUpload() {
        document.getElementById('UploadFile').click();
    }    
	ArrayToColor(v) {
		var result = "#";
		result += this.ValueToByteStr(v[0])
		result += this.ValueToByteStr(v[1])
		result += this.ValueToByteStr(v[2])
		return result;
    }	
    en() {
      //  this.$translate.use("en_US");
    }
    nl() {
        
     //   this.$translate.use("nl_NL");
    }
    GetSchema(name) {
        return this._materialDetails.schema.types[name];
    }
    DownloadMat() {
        var anchor = document.createElement("a");
        anchor.download = this.Name() + ".json";
        anchor.href = this.materialEditorService.MaterialEditorUrl + "download/" + this._materialId;
        anchor.click();
    }
    SetPreprocessorFlags() {
        var matCtrl = this;
        this._materialDetails.values.preprocessorFlags = {};
        this.Macros.forEach(function(macro) {
            matCtrl._materialDetails.values.preprocessorFlags[macro.Name] = macro.Value;
        });
        this.materialEditorService.setMaterialDetails(this._materialId, JSON.stringify(this._materialDetails.values)).then
            ( response => {
                location.reload(true);
            });
    }
    SaveToDevFolder() {
        this.materialEditorService.saveToDevFolder(this._materialId);
    }
    BackToList() {
		this.$state.go('materiallist');
    }

    GetBlendModes() {
        return Object.keys(this._blendFlags);
    }

    get SrcBlend() {
        for( var flag in this._blendFlags ) {
            var gameFlag = this._blendFlags[flag];

            // default to the null flag, note this should be last in the array
            if( this._materialDetails.values.srcBlend == gameFlag )
                return flag;
        }
    }
    set SrcBlend(val) {
        this._materialDetails.values.srcBlend = this._blendFlags[val];
        this.ForceUpdateMaterial();
    }
    get DstBlend() {
        for( var flag in this._blendFlags ) {
            var gameFlag = this._blendFlags[flag];

            // default to the null flag, note this should be last in the array
            if( this._materialDetails.values.dstBlend == gameFlag )
                return flag;
        }
    }
    set DstBlend(val) {
        this._materialDetails.values.dstBlend = this._blendFlags[val];
        this.ForceUpdateMaterial();
    }
}
