angular.module('overlay.qualifyingBox', []).component('qualifyingBoxOverlay', {
    bindings: {
        overlayInfo: '<',
        selectedDriver: '<',
        config: '<',
        customConfig: '<',
        sessionInfo: '<',
        standings: '<',
        standingsHistory: '<',
        entryDataField: '<',
        selectedCarClass: '<',
        selectedEntryInfo: '<',
        cars: '<'
    },
    templateUrl: 'src/components/qualifyingboxoverlay/qualifyingbox.overlay.html',
    controller: function (broadcastService, standingsService, utilService, $interval, $timeout, $scope) {
        var ctrl = this;
        this.carImageSrc = null;
        this.lapTimeCounterInterval = null;
        this.lapTimeCounterValue = null;
        this.lapCompletedFreezeTimeout = null;
        this.sectorCompletedFreezeTimeout = null;

        this.$onChanges = function (changes) {
            if (changes.selectedDriver) {
                var newName = _.get(changes, 'selectedDriver.currentValue.driverName');
                var oldName = _.get(changes, 'selectedDriver.previousValue.driverName');

                if (newName && newName !== oldName) {
                    broadcastService.getDiskOrGameCarImageSrc(
                        changes.selectedDriver.currentValue,
                        _.get(ctrl, 'config.customConfig'),
                        setCarImageSrc
                    );
                }

                var newDriver = _.get(changes, 'selectedDriver.currentValue');
                var oldDriver = _.get(changes, 'selectedDriver.previousValue');

                // Reset counter when needed
                if (newDriver && oldDriver && newDriver.slotID !== oldDriver.slotID) {
                    // Driver changed
                    cancelLapTimeCounterInterval();
                    cancelSectorCompletedFreezeTimeout();
                    cancelLapCompletedFreezeTimeout();
                }
            }

            if (changes.standings) {
                if (!ctrl.selectedDriver) {
                    return;
                }

                var newDriver = _.find(changes.standings.currentValue, { slotID: ctrl.selectedDriver.slotID });
                var oldDriver = _.find(changes.standings.previousValue, { slotID: ctrl.selectedDriver.slotID });
                var newLapStartEventTime = _.get(newDriver, 'lapStartEventTime');
                var oldLapStartEventTime = _.get(oldDriver, 'lapStartEventTime');
                var newSector = _.get(newDriver, 'sector');
                var oldSector = _.get(oldDriver, 'sector');

                if (newLapStartEventTime && !oldLapStartEventTime) {
                    cancelLapTimeCounterInterval();
                } else if (newSector && oldSector && newSector !== oldSector && newSector !== 'SECTOR1') {
                    // Sector changed.
                    cancelLapTimeCounterInterval();
                    ctrl.sectorCompletedFreezeTimeout = $timeout(function () {
                        cancelSectorCompletedFreezeTimeout();
                    }, 5000);
                } else if (newLapStartEventTime && oldLapStartEventTime && newLapStartEventTime !== oldLapStartEventTime) {
                    //  Driver started a new lap (lap start time changed).
                    cancelLapTimeCounterInterval();

                    if (standingsService.isLapTimeValid(_.get(ctrl, 'selectedDriver.lastLapTime'))) {
                        // Freeze lap info for showing the lap time and sector data for the just completed lap.
                        // (Unless last lap was not valid).
                        ctrl.lapCompletedFreezeTimeout = $timeout(function () {
                            cancelLapCompletedFreezeTimeout();
                        }, 10000);
                    }
                } else if (ctrl.selectedDriver.pitting) {
                    cancelLapTimeCounterInterval();
                    return;
                }

                if (!ctrl.lapTimeCounterInterval && newLapStartEventTime) {
                    var lapTimeCounterValue = ctrl.sessionInfo.currentEventTime - newLapStartEventTime;

                    ctrl.lapTimeCounterInterval = $interval(function () {
                        lapTimeCounterValue += 0.1;

                        // Reset counter if it's off by more than one second. (Due to time acceleration in single player
                        // for example).
                        var lapDuration = ctrl.sessionInfo.currentEventTime - newLapStartEventTime;
                        var counterOffBy = Math.abs(lapTimeCounterValue - lapDuration);
                        if (counterOffBy > 1) {
                            lapTimeCounterValue = ctrl.sessionInfo.currentEventTime - newLapStartEventTime;
                        }

                        ctrl.lapTimeCounterValue = lapTimeCounterValue;
                    }, 100);
                }
            }
        }

        function setCarImageSrc(src) {
            if (src) {
                ctrl.carImageSrc = src;
            }
        }

        $scope.$on('$destroy', function () {
            cancelLapTimeCounterInterval();
        });

        function cancelLapTimeCounterInterval() {
            $interval.cancel(ctrl.lapTimeCounterInterval);
            ctrl.lapTimeCounterInterval = null;
            ctrl.lapTimeCounterValue = null;
        }

        function cancelLapCompletedFreezeTimeout() {
            $timeout.cancel(ctrl.lapCompletedFreezeTimeout);
            ctrl.lapCompletedFreezeTimeout = null;
        }

        function cancelSectorCompletedFreezeTimeout() {
            $timeout.cancel(ctrl.sectorCompletedFreezeTimeout);
            ctrl.sectorCompletedFreezeTimeout = null;
        }

        this.getCssCarClassName = function (carClass) {
            return utilService.generateCssCarClassName(carClass);
        }

        this.getBorderLeftStyle = function () {
            return broadcastService.getDriverBorderStyle(ctrl.selectedDriver, ctrl.customConfig, 'borderLeft');
        }

        this.displayTargetLapTime = function () {
            var targetCarClass;
            if (!broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                targetCarClass = _.get(ctrl, 'selectedDriver.carClass');
                if (!targetCarClass) {
                    return '';
                }
            }

            return standingsService.formatDriverLapTime(
                standingsService.findSessionBestLapTimeEntry(ctrl.standings, targetCarClass),
                'bestLapTime',
                3,
                'No time'
            );
        }

        this.getLapTimeCounterClass = function () {
            if (!ctrl.lapCompletedFreezeTimeout || !ctrl.selectedDriver) {
                return null;
            }

            var driverLastLapTime = ctrl.selectedDriver.lastLapTime;
            if (!standingsService.isLapTimeValid(driverLastLapTime)) {
                return null;
            }

            var targetCarClass;
            if (!broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                targetCarClass = _.get(ctrl, 'selectedDriver.carClass');
                if (!targetCarClass) {
                    return null;
                }
            }

            var sessionBestLapTimeInClass = standingsService.findSessionBestLapTime(ctrl.standings, targetCarClass);
            if (!standingsService.isLapTimeValid(sessionBestLapTimeInClass)
                || driverLastLapTime === sessionBestLapTimeInClass
            ) {
                return 'purple';
            }

            var driverBestLapTime = ctrl.selectedDriver.bestLapTime;
            if (!standingsService.isLapTimeValid(driverBestLapTime) || driverLastLapTime === driverBestLapTime) {
                return 'green';
            }

            return null;
        }

        this.displayLapTimeCounterValue = function () {
            if (ctrl.selectedDriver && ctrl.selectedDriver.pitting) {
                return 'Pit';
            }

            if (ctrl.selectedDriver
                && ctrl.selectedDriver.timeIntoLap > 0
                && ctrl.selectedDriver.countLapFlag === 'COUNT_LAP_ONLY'
            ) {
                return 'out lap';
            }

            if (ctrl.sectorCompletedFreezeTimeout) {
                var currentSector = _.get(ctrl, 'selectedDriver.sector');

                if (currentSector) {
                    var currentSectorNumber = currentSector.substr(-1, 1);
                    var sectorTime = standingsService.getDriverSectorTime(
                        ctrl.selectedDriver,
                        currentSectorNumber - 1,
                        'current'
                    );

                    if (standingsService.isLapTimeValid(sectorTime)) {
                        return sectorTime.toFixed(3);
                    }
                }
            } else if (ctrl.lapCompletedFreezeTimeout) {
                return standingsService.formatDriverLapTime(ctrl.selectedDriver, 'lastLapTime');
            }

            var lapTime;

            if (!ctrl.lapTimeCounterValue) {
                if (ctrl.selectedDriver && ctrl.selectedDriver.timeIntoLap > -1) {
                    lapTime = utilService.secToString(ctrl.selectedDriver.timeIntoLap, 1);
                } else if (ctrl.selectedDriver && ctrl.selectedDriver.countLapFlag === 'COUNT_LAP_ONLY') {
                    return 'out lap';
                } else {
                    return '-';
                }
            }

            if (!lapTime) {
                lapTime = utilService.secToString(ctrl.lapTimeCounterValue, 1);
            }

            if (lapTime === '-') {
                return '0.0';
            }

            return lapTime;
        }

        // Display gap to class fastest lap sector (not fastest overall sector)
        this.displaySectorGap = function (sector) {
            if (ctrl.selectedDriver && ctrl.selectedDriver.pitting) {
                var bestLapSectorTime = standingsService.getDriverBestLapSectorTime(ctrl.selectedDriver, sector);

                if (standingsService.isLapTimeValid(bestLapSectorTime)) {
                    return bestLapSectorTime.toFixed(3);
                }

                return '-';
            }

            var propPart = ctrl.lapCompletedFreezeTimeout ? 'last' : 'current';
            var sectorTime = standingsService.getDriverSectorTime(ctrl.selectedDriver, sector, propPart, false);
            if (!standingsService.isLapTimeValid(sectorTime)) {
                return '-';
            }

            var leaderSectorTime = getLeaderSectorTime(sector);
            if (!standingsService.isLapTimeValid(leaderSectorTime)) {
                return sectorTime.toFixed(3);
            }

            var gap = sectorTime - leaderSectorTime;
            if (gap >= 0) {
                return '+' + gap.toFixed(3);
            }

            return gap.toFixed(3);
        }

        function getLeaderSectorTime(sector) {
            var leaderCarClass;
            if (!broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                leaderCarClass = _.get(ctrl, 'selectedDriver.carClass');
            }

            var leader;
            var leaderSectorTime;
            if (ctrl.lapCompletedFreezeTimeout) {
                var sessionBestLapTime = standingsService.findSessionBestLapTime(ctrl.standings, leaderCarClass);
                if (ctrl.selectedDriver && ctrl.selectedDriver.lastLapTime === sessionBestLapTime) {
                    // If driver set the fastest lap of the session, get the sector time from the previous fastest lap
                    // standings to be able to show the correct gap during the lap info freeze.
                    var driverHistory = getDriverHistoryEntries(ctrl.selectedDriver);
                    var ignoreLapEntry = {
                        slotID: ctrl.selectedDriver.slotID,
                        lapIndex: driverHistory.length - 2
                    }

                    var fastestLapEntry = getFastestLapEntry(ctrl.standingsHistory, leaderCarClass, ignoreLapEntry);
                    if (fastestLapEntry) {
                        if (sector === 1) {
                            leaderSectorTime = fastestLapEntry.sectorTime1;
                        } else if (sector === 2) {
                            leaderSectorTime = fastestLapEntry.sectorTime2 - fastestLapEntry.sectorTime1;
                        } else if (sector === 3) {
                            leaderSectorTime = fastestLapEntry.lapTime - fastestLapEntry.sectorTime2;
                        }
                    } else {
                        leaderSectorTime = -1;
                    }
                } else {
                    if (broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                        leader = _.find(ctrl.standings, { position: 1 });
                    } else {
                        leader = _.find(ctrl.standings, { carClass: leaderCarClass, classPosition: 1 });
                    }

                    leaderSectorTime = standingsService.getDriverBestLapSectorTime(leader, sector, false);
                }
            } else {
                if (broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                    leader = _.find(ctrl.standings, { position: 1 });
                } else {
                    leader = _.find(ctrl.standings, { carClass: leaderCarClass, classPosition: 1 });
                }

                leaderSectorTime = standingsService.getDriverBestLapSectorTime(leader, sector, false);
            }

            return leaderSectorTime;
        }

        this.getSectorClass = function (sector) {
            if (!ctrl.selectedDriver) {
                return 'empty';
            }

            var propPart;
            if (standingsService.isDriverPitting(ctrl.selectedDriver)) {
                propPart = 'bestLap';
            } else {
                propPart = ctrl.lapCompletedFreezeTimeout ? 'last' : 'current';
            }

            var driverSectorTime = standingsService.getDriverSectorTime(ctrl.selectedDriver, sector, propPart, false);
            if (driverSectorTime === -1) {
                return 'empty';
            }

            if (standingsService.isDriverPitting(ctrl.selectedDriver)) {
                return 'bestLap';
            }

            if (isPurpleSector(driverSectorTime, sector)) {
                return 'purple';
            }

            // Find the right sector time to compare to before deciding if sector is green or yellow
            var driverBestLapSectorTime;
            if (ctrl.lapCompletedFreezeTimeout) {
                if (standingsService.isLapTimeValid(ctrl.selectedDriver.lastLapTime)
                    && ctrl.selectedDriver.lastLapTime === ctrl.selectedDriver.bestLapTime) {
                    // Driver set their fastest lap on last lap, compare sector times to the previous fastest lap during
                    // lap time freeze
                    var previousBestLapEntry = findDriverFastestLapEntry(ctrl.selectedDriver.lastLapTime);

                    if (previousBestLapEntry) {
                        if (sector === 1) {
                            driverBestLapSectorTime = previousBestLapEntry.sectorTime1;
                        } else if (sector === 2) {
                            driverBestLapSectorTime = previousBestLapEntry.sectorTime2 - previousBestLapEntry.sectorTime1;
                        } else if (sector === 3) {
                            driverBestLapSectorTime = previousBestLapEntry.lapTime - previousBestLapEntry.sectorTime2;
                        }
                    } else {
                        driverBestLapSectorTime = -1;
                    }
                } else {
                    // Just compare to the sector time of the best lap
                    driverBestLapSectorTime = standingsService.getDriverBestLapSectorTime(
                        ctrl.selectedDriver, sector, false
                    );
                }
            } else {
                // During the lap, just compare to the sector time of the best lap
                driverBestLapSectorTime = standingsService.getDriverBestLapSectorTime(
                    ctrl.selectedDriver, sector, false
                );
            }

            if (isGreenSector(driverSectorTime, driverBestLapSectorTime)) {
                return 'green';
            }
            if (isYellowSector(driverSectorTime, driverBestLapSectorTime)) {
                return 'yellow';
            }

            return 'empty';
        }

        function getDriverHistoryEntries(driver) {
            if (!driver || !ctrl.standingsHistory || !ctrl.standingsHistory[driver.slotID]) {
                return [];
            }

            return ctrl.standingsHistory[driver.slotID];
        }

        function getFastestLapEntry(standingsHistory, carClass, ignoreLapEntry = null) {
            var fastestLapEntry = null;

            _.forOwn(standingsHistory, function (driverEntry) {
                _.forEach(driverEntry, function (lapEntry, lapIndex) {
                    if (ignoreLapEntry
                        && lapEntry.slotID === ignoreLapEntry.slotID
                        && lapIndex === ignoreLapEntry.lapIndex) {
                        return; // Continue
                    }

                    if (standingsService.isLapTimeValid(lapEntry.lapTime)
                        && (fastestLapEntry === null || lapEntry.lapTime <= fastestLapEntry.lapTime)
                        && (broadcastService.isMixedClassMode(ctrl.selectedCarClass) || carClass && carClass === lapEntry.carClass)
                    ) {
                        fastestLapEntry = lapEntry;
                    }
                });
            });

            return fastestLapEntry;
        }

        function findDriverFastestLapEntry(ignoreLapTime) {
            try {
                var driverHistory = ctrl.standingsHistory[ctrl.selectedDriver.slotID];
                var fastestLapEntry = null;
                _.forEach(driverHistory, function (lapEntry) {
                    if (standingsService.isLapTimeValid(lapEntry.lapTime)
                        && (fastestLapEntry === null || lapEntry.lapTime <= fastestLapEntry.lapTime)
                        && lapEntry.lapTime !== ignoreLapTime
                    ) {
                        fastestLapEntry = lapEntry;
                    }
                });

                return fastestLapEntry;
            } catch (e) {
                return null;
            }
        }

        // Returns true if driver set the fastest lap in car class on the last completed lap
        function driverSetFastestLap(driver, fastestLapDriver, driverHistory) {
            return driver
                && fastestLapDriver
                && driver.slotID === fastestLapDriver.slotID
                && standingsService.isLapTimeValid(driver.bestLapTime)
                && driverHistory.length > 1
                && driver.bestLapTime === driverHistory[driverHistory.length - 2].lapTime;
        }

        function isPurpleSector(driverSectorTime, sector) {
            var fastestLapCarClass;
            if (!broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                fastestLapCarClass = _.get(ctrl, 'selectedDriver.carClass');
                if (!fastestLapCarClass) {
                    return false;
                }
            }

            var fastestLapDriver = standingsService.findSessionBestLapTimeEntry(ctrl.standings, fastestLapCarClass);

            if (ctrl.lapCompletedFreezeTimeout) {
                var driverHistory = getDriverHistoryEntries(ctrl.selectedDriver);
                var ignoreLapEntry = null;
                if (driverSetFastestLap(ctrl.selectedDriver, fastestLapDriver, driverHistory)) {
                    // Ignore the newly set fastest lap when finding the previous fastest lap entry
                    ignoreLapEntry = {
                        slotID: ctrl.selectedDriver.slotID,
                        lapIndex: driverHistory.length - 2
                    }
                }

                var prevFastestLapEntry = getFastestLapEntry(ctrl.standingsHistory, fastestLapCarClass, ignoreLapEntry);
                if (prevFastestLapEntry) {
                    var previousFastestLapSectorTime;
                    if (sector === 1) {
                        previousFastestLapSectorTime = prevFastestLapEntry.sectorTime1
                    } else if (sector === 2) {
                        previousFastestLapSectorTime = prevFastestLapEntry.sectorTime2 - prevFastestLapEntry.sectorTime1;
                    } else if (sector === 3) {
                        previousFastestLapSectorTime = prevFastestLapEntry.lapTime - prevFastestLapEntry.sectorTime2;
                    }

                    if (driverSectorTime < previousFastestLapSectorTime) {
                        return true;
                    }
                } else {
                    return true;
                }
            } else {
                if (!fastestLapDriver) {
                    // No fastest lap set yet in car class
                    return true;
                }

                var fastestSessionLapSectorTime = standingsService.getDriverBestLapSectorTime(
                    fastestLapDriver,
                    sector,
                    false
                );

                if (standingsService.isLapTimeValid(driverSectorTime)
                    && standingsService.isLapTimeValid(fastestSessionLapSectorTime)
                    && driverSectorTime < fastestSessionLapSectorTime
                ) {
                    return true;
                }
            }

            return false;
        }

        function isGreenSector(driverCurrentSectorTime, driverBestLapSectorTime) {
            return standingsService.isLapTimeValid(driverCurrentSectorTime)
                && ((standingsService.isLapTimeValid(driverBestLapSectorTime)
                    && driverCurrentSectorTime < driverBestLapSectorTime)
                    || !standingsService.isLapTimeValid(driverBestLapSectorTime));
        }

        function isYellowSector(driverCurrentSectorTime, driverBestLapSectorTime) {
            return standingsService.isLapTimeValid(driverCurrentSectorTime)
                && standingsService.isLapTimeValid(driverBestLapSectorTime)
                && driverCurrentSectorTime > driverBestLapSectorTime
        }
    }
});
