angular.module('overlay.standingsTower', []).component('standingsTowerOverlay', {
    bindings: {
        overlayInfo: '<',
        sessionInfo: '<',
        drivers: '<',
        code80: '<',
        customConfig: '<',
        pitTimers: '<',
        pitLaneTimerEnabled: '<',
        pitBoxTimerEnabled: '<',
        entryDataField: '<',
        startingOrder: '<',
        selectedDriver: '<',
        cars: '<'
    },
    templateUrl: 'src/components/standingstoweroverlay/standingstower.overlay.html',
    controller: function (broadcastService, sessionService, standingsService, utilService, $animate, $timeout, $scope) {
        var ctrl = this;

        // Default values that will be overwritten from config.json
        ctrl.config = {
            scrollingListConfig: {
                topRowCount: 3,
                scrollingRowCount: 7
            }
        }

        ctrl.singleCarClassEntries = []; // Used to render data when tower is in specific car class mode
        ctrl.scrollRowsTimeout = null;
        ctrl.scrollingListIndex = ctrl.config.scrollingListConfig.topRowCount;
        ctrl.scrollRowsAnimationDuration = 500; // ms
        ctrl.scrollRowsAnimationDelayPerRow = 30; // ms
        ctrl.animatingPositionChange = false;
        ctrl.overlayEnabled = false;
        ctrl.selectedCarClass = null;

        this.$onInit = function () {
            var towerConfig = _.get(ctrl, 'customConfig.overlays.tower');
            if (towerConfig) {
                ctrl.config = towerConfig;
                ctrl.scrollingListIndex = towerConfig.scrollingListConfig.topRowCount;
            }
        }

        this.$onChanges = function (changes) {
            if (changes.drivers) {
                var newVal = changes.drivers.currentValue;
                var oldVal = changes.drivers.previousValue;

                if (ctrl.overlayInfo) {
                    if (ctrl.overlayInfo.carClass === 'Multiclass') {
                        ctrl.carClasses = _.groupBy(newVal, 'carClass');
                        ctrl.carClasses = _.map(ctrl.carClasses, function(value, key) {
                            return {
                                name: key,
                                entries: value
                            }
                        });
                    } else {
                        if (ctrl.animatingPositionChange) {
                            return;
                        }

                        if (broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                            ctrl.singleCarClassEntries = newVal;
                        } else {
                            ctrl.singleCarClassEntries = _.filter(newVal, { carClass: ctrl.overlayInfo.carClass });
                        }

                        checkScrolling();
                    }
                }

                if (newVal.length !== oldVal.length) {
                    return;
                }

                animatePositionChange(newVal, oldVal);
            }

            if (changes.overlayInfo) {
                if (!changes.overlayInfo.currentValue.visible) {
                    cancelScrollingRows();
                    return;
                }

                ctrl.overlayEnabled = true;
                var newCarClass = _.get(changes, 'overlayInfo.currentValue.carClass');
                var oldCarClass = _.get(changes, 'overlayInfo.previousValue.carClass');
                if (!ctrl.selectedCarClass && newCarClass) {
                    ctrl.selectedCarClass = newCarClass;
                }

                if (newCarClass && oldCarClass && newCarClass !== oldCarClass) {
                    ctrl.carClassChanged = true;

                    $timeout(function() {
                        cancelScrollingRows();
                        ctrl.carClassChanged = false;
                        ctrl.selectedCarClass = newCarClass;
                    }, 2000);
                }
            }
        }

        $scope.$on('$destroy', function() {
            cancelScrollingRows();
        });

        var hideableFieldAnimEnded = false;
        $scope.$on('HIDEABLE_FIELD_ANIM_END', function() {
            if (!hideableFieldAnimEnded) {
                // Timeout to respond to only one of the animation events from the hideable fields
                $timeout(function() {
                    $scope.$broadcast('CONDENSE_TEXT:SCALE_TEXT');
                    hideableFieldAnimEnded = false;
                });
            }

            hideableFieldAnimEnded = true;
        });

        function checkScrolling() {
            var carClass = _.get(ctrl, 'overlayInfo.carClass');
            if (!scrollingIsNeeded() || carClass === 'Multiclass') {
                cancelScrollingRows();
            } else if (ctrl.scrollRowsTimeout === null) {
                startScrollingRows();
            }
        }

        function scrollingIsNeeded() {
            var carClass = _.get(ctrl, 'overlayInfo.carClass');
            var entryCount = _.get(ctrl, 'singleCarClassEntries.length');

            return !broadcastService.isMulticlassClassMode(carClass) && entryCount > getTotalRowCount();
        }

        function cancelScrollingRows() {
            if (ctrl.scrollRowsTimeout !== null) {
                $timeout.cancel(ctrl.scrollRowsTimeout);
                ctrl.scrollRowsTimeout = null;
                ctrl.scrollingListIndex = ctrl.config.scrollingListConfig.topRowCount;
            }
        }

        function startScrollingRows() {
            var scrollingIntervalDelay = _.get(ctrl, 'config.scrollingListConfig.scrollingInterval');
            if (!scrollingIntervalDelay) {
                return;
            }

            ctrl.scrollRowsTimeout = $timeout(function() {
                if (!scrollingIsNeeded()) {
                    cancelScrollingRows();
                    return;
                }

                ctrl.dynamicScrollRowCount = Math.min(
                    ctrl.singleCarClassEntries.length - ctrl.scrollingListIndex,
                    ctrl.config.scrollingListConfig.scrollingRowCount
                );

                $timeout(function() {
                    ctrl.scrollingListIndex = ctrl.getNewScrollListIndex(
                        ctrl.scrollingListIndex,
                        ctrl.singleCarClassEntries
                    );

                    $timeout(function() {
                        startScrollingRows();
                    }, getTotalScrollRowsAnimationDuration());
                });
            }, scrollingIntervalDelay);
        }

        // Get how long it will take to get through the whole scrolling animation (with the delay effect for each row).
        function getTotalScrollRowsAnimationDuration() {
            return ctrl.scrollRowsAnimationDuration + ctrl.dynamicScrollRowCount * ctrl.scrollRowsAnimationDelayPerRow;
        }

        // Returns the total number of rows in the tower (fixed + scrolling).
        function getTotalRowCount() {
            var fixedRowCount = getFixedRowCount();
            var scrollingRowCount = getScrollingRowCount();

            if (_.isNil(fixedRowCount) || _.isNil(scrollingRowCount)) {
                return null;
            }

            return fixedRowCount + scrollingRowCount;
        }

        // Returns the number of fixed (top) rows. (When the tower is in the "class" mode).
        function getFixedRowCount() {
            return _.get(ctrl, 'config.scrollingListConfig.topRowCount');
        }

        // Returns the number of scrolling rows. (When the tower is in the "class" mode).
        function getScrollingRowCount() {
            return _.get(ctrl, 'config.scrollingListConfig.scrollingRowCount');
        }

        this.getNewScrollListIndex = function(currentScrollListIndex, currentCarClassEntries) {
            var newScrollListIndex = currentScrollListIndex + ctrl.config.scrollingListConfig.scrollingRowCount;
            if (!currentCarClassEntries || newScrollListIndex > currentCarClassEntries.length - 1) {
                newScrollListIndex = ctrl.config.scrollingListConfig.topRowCount;
            }

            return newScrollListIndex;
        }

        this.getRowClass = function(entry, first, index) {
            return [
                {
                    ['slot-' + entry.slotID]: entry && entry.slotID > -1,
                    first: first,
                    'entry-faded-out': ctrl.getFadedOut(entry),
                    'last-fixed-row': _.get(ctrl, 'config.scrollingListConfig.topRowCount') === index + 1,
                    'first-scrolling-row': _.get(ctrl, 'config.scrollingListConfig.topRowCount') + 1 === index + 1,
                    'selected-driver': _.get(ctrl, 'selectedDriver.slotID') === entry.slotID,
                    'has-drs-active': standingsService.hasDrsActive(entry)
                },
                ctrl.getCssCarClassName(entry.carClass)
            ]
        }

        this.getCssCarClassName = function(carClass) {
            return utilService.generateCssCarClassName(carClass);
        }

        this.getStandingsGroupHeading = function(carClassName) {
            const carClass = carClassName === 'GT3' ? 'LMGT3' : carClassName;
            const dynamicData = _.get(ctrl, 'overlayInfo.dynamicData');

            switch (dynamicData) {
                case 'gapToLeader':
                    return `${carClass} - Gap`;
                case 'gapToNext':
                    return `${carClass} - Interval`;
                case 'pitStopCount':
                    return `${carClass} - Stops`;
                case 'positionChange':
                    return `${carClass} - Pos`;
                case 'energyRemaining':
                    return `${carClass} - NRG`;
                case 'tireName':
                    return `${carClass} - Tyre`;
                default:
                    return `${carClass}`;
            }
        }

        this.getStandingsGroupBorderStyle = function(carClassName) {
            var classColor = broadcastService.getCarClassColorFromConfig(ctrl.customConfig, carClassName);

            if (!classColor) {
                return null;
            }

            return {
                borderLeft: 'solid 0.2em ' + classColor,
                transform: ctrl.overlayInfo.hideField === 'field2' ? 'translateX(-2rem)' : null
            }
        }

        this.getEntryPitDuration = function(entry, propName) {
            var pitStopEntry = _.find(ctrl.pitTimers, { slotID: entry.slotID });

            if (!pitStopEntry) {
                return null;
            }

            return pitStopEntry[propName];
        }

        this.getSwipeStyle = function(carClass) {
            if (carClass === 'Multiclass') {
                return {
                    backgroundColor: broadcastService.getCustomConfigSeriesColor(ctrl.customConfig)
                }
            }

            var carClassStyle = broadcastService.getCarClassColorStyle(
                { carClass: carClass },
                ctrl.customConfig,
                'backgroundColor'
            );

            if (!carClassStyle) {
                return {
                    backgroundColor: broadcastService.getCustomConfigSeriesColor(ctrl.customConfig)
                }
            }

            return carClassStyle;
        }

        this.isRaceSession = function () {
            return sessionService.isRaceSession(this.sessionInfo);
        }

        this.displaySessionName = function () {
            return sessionService.displaySessionName(this.sessionInfo);
        }

        this.displaySessionDuration = function () {
            return sessionService.displaySessionDuration(this.sessionInfo, this.drivers);
        }

        this.sessionBestLapTime = () => {
            return standingsService.findSessionBestLapTime(ctrl.drivers);
        }

        this.getPositionStyle = function() {
            var bgColor = utilService.getCustomColor(
                'position',
                'backgroundColor',
                ctrl.customConfig
            );
            var color = utilService.getCustomColor('position', 'color', ctrl.customConfig);

            return '--custom-position-background-color: ' + bgColor + '; --custom-position-color: ' + color;
        }

        ctrl.getFadedOut = function(driver) {
            return standingsService.getEntryFadedOut(driver, ctrl.sessionInfo);
        }

        function getSelectedCarClass() {
            return _.get(ctrl, 'overlayInfo.carClass');
        }

        function animatePositionChange(newVal, oldVal) {
            _.forEach(newVal, (newEntry) => {
                var oldEntry = _.find(oldVal, (oldEntry) => {
                    return newEntry.slotID === oldEntry.slotID;
                });

                var positionProp = broadcastService.isMixedClassMode(getSelectedCarClass())
                    ? 'position'
                    : 'classPosition';
                if (oldEntry && oldEntry[positionProp] !== newEntry[positionProp]) {
                    $timeout(function() {
                        var elem = document.querySelector('.standings-tower-overlay .slot-' + newEntry.slotID);

                        if (!elem) {
                            return;
                        }

                        var moveAmount;

                        if (newEntry[positionProp] < oldEntry[positionProp]) {
                            moveAmount = (oldEntry[positionProp] - newEntry[positionProp]) * 100;
                        } else {
                            moveAmount = ((newEntry[positionProp] - oldEntry[positionProp]) * 100) * -1;
                        }

                        ctrl.animatingPositionChange = true;
                        elem.style.transform = 'translateY(' + moveAmount + '%)';

                        $timeout(() => {
                            elem.style.transition = 'transform 500ms';
                            elem.style.transform = 'translateY(0)';

                            $timeout(function() {
                                elem.style.transition = null;
                                ctrl.animatingPositionChange = false;
                            }, 600)
                        }, 0);
                    })
                }
            });
        }
    }
}).filter('singleClassMode', function () {
    return (items, config, scrollingListIndex) => {
        var topRowCount = _.get(config, 'scrollingListConfig.topRowCount');
        var scrollingRowCount = _.get(config, 'scrollingListConfig.scrollingRowCount');
        var filtered = items.slice(0, topRowCount);
        var scrollingRows = items.slice(scrollingListIndex, scrollingListIndex + scrollingRowCount);

        return filtered.concat(scrollingRows);
    }
}).directive('hideableField', function($rootScope) {
    return {
        link: function($scope, $elem) {
            $elem[0].addEventListener('animationend', function() {
                $rootScope.$broadcast('HIDEABLE_FIELD_ANIM_END');
            });

            $elem[0].addEventListener('transitionend', function() {
                $rootScope.$broadcast('HIDEABLE_FIELD_ANIM_END');
            });
        }
    }
}).directive('swipeWrapper', function() {
    return {
        link: function($scope, $elem) {
            var elem = $elem[0];
            elem.addEventListener('animationstart', () => {
                // Set a custom CSS property when swipe animation starts. It's used in the CSS animation to set the
                // clip-path polygon shape to skew the left edge of the color swipe.
                var customCssPropName = '--swipe-skew-side-length';
                var skewLength = getSwipeSkewSideLength();
                elem.style.setProperty(customCssPropName, skewLength + 'px');
            });

            function getSwipeSkewSideLength() {
                var towerElem = document.querySelector('.standings-tower-overlay');
                if (towerElem) {
                    var skewAngle = 22;
                    return Math.round(Math.tan(skewAngle * Math.PI / 180) * towerElem.clientHeight);
                }
            }
        }
    }
});
