var liveTimingApp = angular.module('liveTimingApp', [
    'BroadcastService',
    'SessionService',
    'StandingsService',
    'UtilService',
    'VehicleInfoService',
    'controlpanel.cameraControl',
    'rf2.class-position-badge',
    'rf2.fallbackImage'
]);

liveTimingApp.controller('LiveTimingController', function(
    broadcastService,
    sessionService,
    standingsService,
    utilService,
    vehicleInfoService,
    $scope,
    $interval
) {
    var ctrl = this;
    var wsUrl = 'ws://' + location.hostname + ':' + (Number(location.port) + 1) + '/websocket/controlpanel';
    var ws;
    var uiWsUrl = 'ws://' + location.hostname + ':' + (Number(location.port) + 1) + '/websocket/ui';
    var uiWs;
    ctrl.sessionInfo = null;
    ctrl.standings = [];
    ctrl.liveStandings = []; // For now, use tire compound data from the UI LiveStandings WS.
    ctrl.standingsDebounced = [];
    ctrl.standingsHistory = null;
    ctrl.carClassGroups = null;
    ctrl.carClasses = [];
    ctrl.removeColumns = ['battleIndicator', 'cameras'];
    ctrl.overlays = null;
    ctrl.config = {};
    ctrl.customConfig = null;
    ctrl.vehicleClassesInfo = null;
    ctrl.dynamicVehicleClassColors = [];
    ctrl.tableConfig = {
        liveTimingPage: true,
        classPositionBadge: true,
        carImage: true,
        tire: true
    };
    ctrl.displaySettings = null;

    broadcastService.getSessionInfo().then(function(response) {
        ctrl.sessionInfo = response.data;
    });

    broadcastService.getStandings().then(function(response) {
        ctrl.standings = response.data;
        ctrl.standings = _.sortBy(ctrl.standings, 'position');
        ctrl.carClasses = _.uniq(_.map(ctrl.standings, 'carClass'));

        addVehicleClassColorProps(ctrl.standings);
    });

    broadcastService.getStandingsHistory().then(function(response) {
        ctrl.standingsHistory = response.data;
    });

    broadcastService.getVehicleClassesInfo().then((info ) => {
        ctrl.vehicleClassesInfo = info;

        addVehicleClassColorProps(ctrl.standings);
    });

    broadcastService.getDisplayOptions().then(function(displaySettings) {
        ctrl.displaySettings = displaySettings;
    });

    getOverlays(true).then(() => {
        broadcastService.getCustomConfig(ctrl.config.customConfig).then((customConfig) => {
            ctrl.customConfig = customConfig;
        });
    });

    initWebSocket();
    initUiWebSocket(); // For now, use tire compound data from the UI LiveStandings WS.

    $interval(function() {
        // Check that we have a WebSocket connection
        if (!ws || ws.readyState === 3) {
            initWebSocket();
        }
    }, 5000);

    $interval(() => {
        getOverlays();
    }, 1000);

    function getOverlays(canUpdateVehicleOverrideInfo = false) {
        return broadcastService.getOverlays().then(function(response) {
            ctrl.overlays = response.data;

            if (response.data.selectedCustomOverlay) {
                ctrl.config.customConfig = response.data.selectedCustomOverlay;

                if (canUpdateVehicleOverrideInfo) {
                    vehicleInfoService.getVehicleOverrideInfo(response.data.selectedCustomOverlay).then((overrideInfo) => {
                        vehicleInfoService.setVehicleOverrideInfo(overrideInfo);
                    }).catch((e) => {
                        vehicleInfoService.setVehicleOverrideInfo(null);
                    });
                }
            }
        }).catch(() => {
            if (!ctrl.overlays) {
                // The control panel page usually sets overlay data. Fetch it in case we're using live timing standalone.
                broadcastService.getConfig().then((config) => {
                    ctrl.config = config;

                    broadcastService.getCustomConfig(config.customConfig).then((customConfig) => {
                        ctrl.customConfig = customConfig;
                        ctrl.overlays = customConfig.overlays;
                        ctrl.overlays.selectedCustomOverlay = config.customConfig;

                        broadcastService.saveOverlays(customConfig.overlays);
                    });
                });
            }
        });
    }

    function initWebSocket() {
        ws = new WebSocket(wsUrl);

        ws.onmessage = function(event) {
            var message = JSON.parse(event.data);

            if (message && message.type === 'standings') {
                ctrl.standings = message.body;
                ctrl.standings = _.sortBy(ctrl.standings, 'position');
                ctrl.carClasses = _.uniq(_.map(ctrl.standings, 'carClass'));

                addVehicleClassColorProps(ctrl.standings);

                // For now, use tire compound data from the UI LiveStandings WS.
                if (ctrl.liveStandings) {
                    _.forEach(ctrl.liveStandings, liveStandingsEntry => {
                        const standingsEntry = ctrl.standings.find(e => e.slotID === liveStandingsEntry.slotID);

                        standingsEntry.tireCompoundNameFront = liveStandingsEntry.tireCompoundNameFront;
                        standingsEntry.tireCompoundNameRear = liveStandingsEntry.tireCompoundNameRear;
                    });
                }
            }

            if (message && message.type === 'sessionInfo') {
                ctrl.sessionInfo = message.body;
            }

            if (message && message.type === 'standingsHistory') {
                ctrl.standingsHistory = message.body;
            }

            updateDataProps();
        }
    }

    // For now, use tire compound data from the UI LiveStandings WS.
    function initUiWebSocket() {
        uiWs = new WebSocket(uiWsUrl);

        uiWs.onopen = function() {
            const subMsg = {
                messageType: 'SUB',
                topic: 'LiveStandings'
            }

            uiWs.send(JSON.stringify(subMsg));

            uiWs.onmessage = function(event) {
                var message = JSON.parse(event.data);

                if (message && message.topic === 'LiveStandings') {
                    ctrl.liveStandings = message.body;
                }
            }
        }
    }

    function updateDataProps() {
        if (!ctrl.sessionInfo || !ctrl.standings || ctrl.standings.length === 0) {
            return;
        }

        ctrl.carClassGroups = broadcastService.getCarClassGroups(ctrl.standings);

        if (ctrl.standingsHistory) {
            addSessionPurpleSectors(ctrl.sessionInfo, ctrl.standingsHistory, ctrl.carClassGroups);
            addSectorTimeInfo(ctrl.sessionInfo, ctrl.standings, ctrl.standingsHistory);
        }

        broadcastService.addStandingsProps(ctrl.standings, ctrl.carClassGroups);

        let vehicleClassesWithConfiguredColors = Object.keys(ctrl.customConfig?.colors?.carClasses || {}).map(carClass => {
            return {
                vehicleClass: carClass
            }
        });

        ctrl.dynamicVehicleClassColors = broadcastService.getDynamicVehicleClassColors(
            broadcastService.findVehicleClassesNeedingDynamicColor(ctrl.standings, vehicleClassesWithConfiguredColors)
        );
    }

    function addVehicleClassColorProps(standings) {
        _.forEach(standings, entry => {
            let vehicleClassColorConfig = ctrl.customConfig?.colors?.carClasses?.[entry.carClass];

            if (vehicleClassColorConfig) {
                const bgColor = vehicleClassColorConfig.backgroundColor;
                const textColor = vehicleClassColorConfig.color;

                if (utilService.validateHexColor(bgColor)) {
                    entry.vehicleClassColor = bgColor;
                }

                if (utilService.validateHexColor(textColor)) {
                    entry.vehicleClassTextColor = textColor;
                }
            }
        });
    }

    function addSessionPurpleSectors(sessionInfo, standingsHistory, carClassGroups) {
        if (!sessionInfo || !standingsHistory || !carClassGroups) {
            return;
        }

        sessionInfo.lapTimeInfo = {
            purpleSectors: standingsService.getSessionFastestSectors(standingsHistory),
            purpleSectorsInClass: standingsService.getSessionFastestSectorsInClass(standingsHistory, carClassGroups)
        }
    }

    function addSectorTimeInfo(sessionInfo, standings, history) {
        if (!sessionInfo || !history) {
            return;
        }

        // Add driver lap time (and sector time) info
        _.forEach(standings, function(entry) {
            entry.lapTimeInfo = {};

            for (var sector = 1; sector <= 3; sector++) {
                entry.lapTimeInfo['bestSector' + sector] = standingsService
                    .findBestSectorTime(history[entry.slotID], sector);

                var overallPurpleTime = _.get(
                    sessionInfo,
                    'sessionInfo.lapTimeInfo.purpleSectors.' + sector + '.time'
                );

                if (overallPurpleTime && entry.lapTimeInfo['bestSector' + sector] === overallPurpleTime) {
                    entry.lapTimeInfo['hasPurpleSector' + sector] = true;
                }

                var carClassPurpleTime = _.get(
                    sessionInfo,
                    'lapTimeInfo.purpleSectorsInClass.' + [entry.carClass] + '.' + [sector] + '.time'
                );

                if (carClassPurpleTime && entry.lapTimeInfo['bestSector' + sector] === carClassPurpleTime) {
                    entry.lapTimeInfo['hasPurpleSector' + sector + 'InClass'] = true;
                } else {
                    entry.lapTimeInfo['hasPurpleSector' + sector + 'InClass'] = false;
                }
            }
        });
    }

    ctrl.displaySessionName = function() {
        return sessionService.displaySessionName(ctrl.sessionInfo);
    }

    ctrl.getSessionTimeText = function() {
        return sessionService.displaySessionDuration(this.sessionInfo, this.standings);
    }
});
